package loquebot.drives;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.AddAmmo;
import cz.cuni.pogamut.MessageObjects.ChangedWeapon;

import loquebot.Main;
import loquebot.util.LoqueListener;
import loquebot.util.LoqueWeaponInfo;
import loquebot.memory.LoqueMemory;
import loquebot.body.LoqueTravel;
import loquebot.body.LoqueRunner;

/**
 * Responsible for handling bloody combat.
 *
 * <p>Main purpose of this drive is to decide, whether a combat is to be dealt
 * with and resolves the combat upon oportunity. The combat includes choosing
 * of enemy; choosing of weapon to fight with; calculating the best strafing
 * locations; reasonable aiming at the target; and also foraging nearby items
 * while still engaging the enemy.</p>
 *
 * <h4>Strafing and foraging</h4>
 *
 * <p>The strafing caluculations are supposed to determine locations to where
 * to strafe to in order to dodge enemy fire and keep reasonable distance from
 * the enemy at the same time. This resembles experienced human player. Current
 * weapon is considered (e.g. shield gun needs a close range, rocket launcher
 * prefers some distance).</p>
 *
 * <p>Also, nearby vials, ammo and armor may be considered, if they are easily
 * reachable. However, foraging should not disturb main strafing behaviour. It
 * should only enhance it since the strafing can really make a difference and
 * save the day.</p>
 *
 * <h4>Aiming</h4>
 *
 * <p>Some weapons spawn flying <i>projectiles</i> instead of speeding bullets
 * and these projectiles are distance dependent. Themain idea is that when our
 * rocket projectile finally reaches its target, the target has already moved
 * forward and therefore. If always aiming at the current enemy location, our
 * projectile would always miss, unless the enemy would stand still.</p>
 *
 * <p>Another problem arises from the time lag rising between the server and
 * our logic. Should we tell the server to aim at the current enemy location,
 * by the time our message arives to the server, the enmy has already moved
 * a few inches.</p>
 *
 * <p>Therefore, for slow projectiles we try to calculate aim-ahead locations
 * based on current enemy velocity and projectile speed. Triangle is calculated
 * between agent location, enemy location and projected collision of enemy with
 * the projectile. Law of sines can predict location of their intersection.</p>
 *
 * <p> For speeding bullets, the aim-ahead location is calculated as a static
 * value based on the current direction of the enemy velocity. This way, we
 * try to <i>aim at the enemy shoulder instead of enemy heart</i>. There is no
 * difference in damage, but it helps compensate for the time lag.</p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueCombat extends LoqueCombatBase
{
    /**
     * Player that we recently tried to tear up. We always try mangle the same
     * enemy for as long as possible to prevent fighting with more enemies at
     * the same time. Doing so would decrease bot's combat fruitfulness.
     */
    private Player lastTarget = null;

    /**
     * Whether to reset all combat info.
     * Used in case of agent's or enemy death.
     */
    private boolean killCombat = false;

    /*========================================================================*/

    /**
     * Main logic of the combat. Drives the choosing of enemy and the combat.
     *
     * <h4>Cook book</h4>
     * <ul>
     * <li>Choose a target, if we have no target yet. When a new target is set,
     * try to attack it asap. When no target can be chosen, disengage and look
     * around for other imminent danger (listen for noises, turn around).</li>
     * <li>If we have a recent target, refresh its current location. If that
     * target got lost from sight (run away or got killed), disengage and look
     * around for other danger.</li>
     * <li>Attack the recent target.</li>
     * </ul>
     *
     * @return True, if the drive decided what shall we do now. False otherwise.
     */
    @Override public boolean doLogic ()
    {
        // are we alowed to operate?
        if (!main.optionCombat)
            return false;

        // include combat logs?
        enableLogs (main.optionCombatLogs);

        // do we kill current combat?
        if (killCombat)
        {
            // reset info
            lastTarget = null;
            // reset flag
            killCombat = false;
        }

        // consider change of target..
        if (!chooseTarget (memory.players.getVisibleEnemies ()))
        {
            // disengage upon no enemies to shoot
            disengage ();
            // look for possible danger
            return reassure ();
        }

        // now, tear the target up..
        return attack (false);
    }

    /*========================================================================*/

    /**
     * Tries to attact given player. Includes aiming, strafing and foraging.
     *
     * <h4>Pogamut troubles</h4>
     * 
     * Fast dodging would be great during combats.. However, right now, the
     * pogamut platform does not seem to handle dodges correctly enough. The
     * agent either keeps colliding, or starts flying around. Therefore, we'll
     * rather keep it simple and stay on the ground.
     *
     * @param force Whether there is no way to escape the combat.
     * @return Returns true, if attacking successfully. False upon errors.
     */
    private boolean attack (boolean force)
    {
        // hovno, hovno, zlata rybka
        if (lastTarget == null)
        {
            // no action was taken
            return false;
        }

        // do we have at least something to fight with?
        if (!memory.inventory.hasLoadedWeapon ())
        {
            // fight at all costs?
            if (!force)
            {
                // try to resolve the combat pacefully
                return reassure ();
            }

            // well, the only gun left, the shield gun needs a close range
            if (!lastTarget.reachable)
                // don't sit around and wait for your toll
                return false;
        }

        // cheat: keep current weapon full
        if (main.optionCombatCheatAmmo)
            cheatRefillAmmo ();

        // choose (or check) the weapon for this combat
        checkWeaponEfficiency (lastTarget);

        // compute new agent focal point (to where to aim to)
        // note: resolves aim-ahead computations
        Triple newAgentFocalPoint = getAimWeaponLocation (lastTarget);

        // check where we're strafing, beware collisions, etc.
        checkStrafingDirection (lastTarget);

        // start shooting..
        checkWeaponShooting (lastTarget, newAgentFocalPoint);

        // decide, whether to jump ro to strafe
        if (Math.random ()  < (1 / (main.logicFrequency * 1.6)))
        {
            // look at the new focal point..
            body.turnToLocation(newAgentFocalPoint);

            // are we going to jump or double jump?
            if (Math.random () < .6)
            {
                body.jump ();
                log.fine ("Combat.attack(): jumping");
            }
            else
            {
                body.doubleJump ();
                log.fine ("Combat.attack(): double jumping");
            }
        }
        else
        {
            // compute new agent location (to where to run to)
            // note: resolves strafing and pick-ups
            Triple newAgentLocation = getStrafeAroundLocation (lastTarget);

            // strafe to the new location and look at the new focal point..
            runner.runToLocation(newAgentLocation, newAgentFocalPoint, false);
        }

        // an action was taken
        return true;
    }

    /**
     * Disengages from all fighting and stops shooting if necessary.
     *
     * <p>This method is called when no enemy is on sight for apparent reason:
     * to stop wasting ammo and to prepare for the next combat.</p>
     *
     * @return Returns true, if an action was taken. False otherwise.
     */
    private boolean disengage ()
    {
        // still shooting something?
        if (memory.self.isShooting ())
        {
            // if the weapon is chargeable, keep it charged
            if (currentWeaponInfo.priKeepShooting)
            {
                // but only, when there is a reason to do that..
                if (
                    (memory.senses.isBeingDamaged ())
                    || (memory.senses.seeIncomingProjectile ())
                    || (memory.senses.seeEnemiesNearby ())
                )
                {
                    log.fine ("Combat.disengage(): disengaging, but keep shooting");

                    // no action was taken
                    return false;
                }
            }

            log.fine ("Combat.disengage(): disengaging, stop shooting");

            // stop wasting ammo
            body.stopShoot ();

            // no action was taken
            return false;
        }

        // nothing to disengage from
        // no action was taken
        return false;
    }

    /**
     * Tries to verify that there is no danger around. Listens to noises,
     * checks projectiles, turns around and other similar things.
     *
     * <p>This method is called when we loose recent target and there is noone
     * visible around. We turn around to make sure we're clear.</p>
     *
     * <h4>Future</h4>
     * 
     * This method should consider means of the best escape instead of simply
     * ignoring the target on sight. This might include using the shield gun
     * to protect us from gunfire and choosing an item that is away from the
     * enemy.
     *
     * @return Returns true, if we took some action. False otherwise.
     */
    private boolean reassure ()
    {
        // was there recently something we should worry about?
        if (
            // NOTE: Unless bot can safely distinct noises made all by himself
            // from noises made by enemies, we can not rely on any noise senses.
            // Otherwise the bot would panic each time he opens a door or picks
            // something up.
            (memory.senses.isBeingDamaged ())
            || (memory.senses.seeIncomingProjectile ())
            || (memory.senses.seeEnemiesNearby ())
        )
        {
            // do we see any enemy?
            if (!memory.players.hasVisibleEnemies ())
            {
                // turn around to find some..
                log.fine ("Combat.reassure(): can't see the danger.. turning around");
                body.turnHorizontal(strafingRight ? 180 : -180);
                return true;
            }

            // yes, there is an enemy on sight?
            log.fine ("Combat.reassure(): imminent danger on sight!");
            return false; //attack (true);
        }

        // no apparent danger around
        // no action was taken
        return false;
    }

    /*========================================================================*/

    /**
     * Considers, whether a change of target is appropriate.
     * Chooses a new target from the given list of players.
     *
     * <p>This method tries to pick an enemy from the given list, considering
     * the weaponry of the enemies and their distance.</p>
     *
     * @param enemies List of players to choose from.
     * @return Returns true, if a new target is acquired. False otherwise.
     */
    private boolean chooseTarget (ArrayList<Player> enemies)
    {
        // hovno, hovno, zlata rybka
        if (enemies == null)
            return (lastTarget != null);

        // don't we have a target already?
        if (lastTarget != null)
        {
            // where's that target now?
            Player enemy = memory.players.getVisiblePlayer (lastTarget.UnrealID);
            // did we loose the target?
            if (enemy == null)
            {
                log.info ("Combat.chooseTarget(): lost target " + lastTarget.UnrealID);

                // should we desire to pursue
                pursue.setTarget(lastTarget);
            }

            // refresh the recent target
            lastTarget = enemy;
        }

        // calculate the distance of recent target
        double lastTargetDistance = (lastTarget != null)
            ? memory.self.getSpaceDistance(lastTarget.location) : 0;

        // currently chosen enemy
        Player chosen = null;

        // run through the given enemies..
        for (Player enemy: enemies)
        {
            // if we have a target already
            if (lastTarget != null)
            {
                // check, whether the new target is even worth the change
                if (
                    // is the recent target at comparable distance?
                    (lastTargetDistance - 1000) < memory.self.getSpaceDistance(enemy.location)
                    // is the recent target at comparable height?
                    && (
                        // is it comparably below the agent
                        ((lastTarget.location.z - 100) < memory.self.getLocation ().z)
                        // is it comparably below the new target
                        || ((lastTarget.location.z - 100) < enemy.location.z)
                    )
                )
                {
                    // then keep fighting the recent target..
                    break;
                }
            }

            // do we have a chosen one?
            if (chosen == null)
            {
                // set first enemy as chosen one..
                chosen = enemy;
            }
            else
            {
                double chosenDistance = memory.self.getSpaceDistance(chosen.location);
                double enemyDistance = memory.self.getSpaceDistance(enemy.location);

                // long-range-compare their distance..
                if ((chosenDistance - 600) > enemyDistance)
                {
                    // close enemies pose greate danger,
                    // get rid of them sooner rather than later
                    chosen = enemy;
                    break;
                }
                // long-range-compare their distance..
                if ((chosenDistance + 600) < enemyDistance)
                {
                    // close enemies pose greate danger,
                    // get rid of them sooner rather than later
                    break;
                }

                // approx-compare their z-axis
                if ((chosen.location.z - 50) > enemy.location.z)
                {
                    // it is always better to fight with someone below,
                    // espcially with a rocket launcher..
                    chosen = enemy;
                    break;
                }
                // approx-compare their z-axis
                if ((chosen.location.z + 50) < enemy.location.z)
                {
                    // it is always better to fight with someone below,
                    // espcially with a rocket launcher..
                    break;
                }

                // approx-compare their distance..
                if ((chosenDistance - 300) > enemyDistance)
                {
                    // closer enemies pose greater danger,
                    // get rid of them sooner rather than later
                    chosen = enemy;
                    break;
                }
                // approx-compare their distance..
                if ((chosenDistance + 300) < enemyDistance)
                {
                    // closer enemies pose greater danger,
                    // get rid of them sooner rather than later
                    break;
                }

                // approx-compare their weapons..
                if (compareEnemyWeapons (chosen.weapon, enemy.weapon) < 100)
                {
                    // disturb the mighty ones
                    chosen = enemy;
                    break;
                }
                // approx-compare their weapons..
                if (compareEnemyWeapons (chosen.weapon, enemy.weapon) > 100)
                {
                    // disturb the mighty ones
                    break;
                }

                // could not decide reasonably, which one is better
                if (Math.random () < .5)
                {
                    // therefore, decide by random choice
                    chosen = enemy;
                    break;
                }
            }
        }

        // did we choose a target?
        if (chosen != null)
        {
            if (lastTarget != null)
                log.info ("Combat.chooseTarget(): loosen recent target " + lastTarget.UnrealID);

            // list possible targets to log..
            for (Player enemy: enemies)
            {
                log.finest (
                    "Combat.chooseTarget(): possible target:" + enemy.UnrealID
                    + ", z-location " + enemy.location.z
                    + ", distance " + memory.self.getSpaceDistance(enemy.location)
                    + ", weapon " + enemy.weapon
                );
            }

            log.info ("Combat.chooseTarget(): setting new combat target " + chosen.UnrealID);

            // set new target
            lastTarget = chosen;

            // increment combat counter..
            memory.self.incTotalCombats ();

            // pick new strafing direction
            strafingRight = (Math.random () > .5);

            // kill current travel ticket
            travel.killTicket ();

            // and report succcess
            return true;
        }

        // report failure based on recent target..
        return (lastTarget != null);
    }

    /*========================================================================*/

    /**
     * Issues a new deccission about which weapon to use in combat.
     *
     * <p>This method is called from outside of combat, e.g. upon weapon or
     * ammo pick-up, etc. The real decision is then made during main combat
     * logic.</p>
     *
     * @param weapon Weapon to which to rearm to. May be null!
     */
    private void considerRearm (AddWeapon weapon)
    {
        // do we have a weapon to consider?
        if (weapon == null)
            return;

        // retreive the current weapon
        AddWeapon current = memory.inventory.getCurrentWeapon ();

        // if we're holding some small weapon which we want to forget about
        if (
            (current == null)
            || LoqueWeaponInfo.getInfo(current.weaponType).minorWeapon
        )
        {
            // and we're fighting something already
            if (lastTarget != null)
            {
                // invoke a new decission about what weapon is to be used..
                log.fine ("Combat.considerRearm(): invoke rearm decission");
                considerRearm = true;
            }
            // otherwise, since we're not fighting anything right now, do the
            // weapon decission manually..
            else checkWeaponEfficiency (null);
        }
    }

    /*========================================================================*/

    /**
     * Kills all combat info. Used in case of agent's death or enemy death.
     */
    private void killCombat ()
    {
        // setup kill flag
        killCombat = true;
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            // reset combat info
            killCombat ();
        }

        /**
         * Agent picked up an ammo pack.
         * @param msg Message to handle.
         */
        private void msgAddAmmo (AddAmmo msg)
        {
            // consider change of weapon
            considerRearm (memory.inventory.getWeapon(msg.weaponType));
        }

        /**
         * Agent picked up a weapon.
         * @param msg Message to handle.
         */
        private void msgAddWeapon (AddWeapon msg)
        {
            // did we picked-up a gun we already carry?
            if (msg.ID != 0)
            {
                // consider change of weapon
                considerRearm (msg);
            }
        }

        /**
         * Agent changed weapon in hand.
         * @param msg Message to handle.
         */
        private void msgChangedWeapon (ChangedWeapon msg)
        {
            // log the change
            log.config ("Combat.Listener: changed to " + msg.cls);
        }

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case ADD_WEAPON:
                    msgAddWeapon ((AddWeapon) msg);
                    return;
                case ADD_AMMO:
                    msgAddAmmo ((AddAmmo) msg);
                    return;
                case CHANGED_WEAPON:
                    msgChangedWeapon ((ChangedWeapon) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener(this, MessageType.SPAWN);
            body.addTypedRcvMsgListener(this, MessageType.ADD_WEAPON);
            body.addTypedRcvMsgListener(this, MessageType.ADD_AMMO);
            body.addTypedRcvMsgListener(this, MessageType.CHANGED_WEAPON);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /**
     * Loque Runner.
     */
    private LoqueRunner runner;

    /*========================================================================*/

    /** Loque pursue. */
    protected LoquePursue pursue;
    /** Loque pursue. */
    protected LoqueTravel travel;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     * @param travel Loque travel.
     * @param pursue Loque pursue.
     */
    public LoqueCombat (Main main, LoqueMemory memory, LoqueTravel travel, LoquePursue pursue)
    {
        super (main, memory);
        this.travel = travel;
        this.pursue = pursue;

        // create runner object
        this.runner = new LoqueRunner (main, memory);

        // create listener
        this.listener = new Listener ();
    }
}